#!/bin/bash

set -euo pipefail

LOGFILE="/var/log/kiosk-install.log"
exec > >(tee -a "$LOGFILE") 2>&1

# Track current step for error reporting
current_step="initial setup"

# Trap error and show message
trap 'echo -e "\n❌ Installation failed during: $current_step\nCheck the log at $LOGFILE for details." >&2' ERR

function section() {
    current_step="$1"
    echo -e "\n🛠️ === $current_step ==="
}

function addOrReplaceCronJob() {
    if test -f "/var/spool/cron/crontabs/$4"; then
        echo "Crontab exists for $4"
    else
        touch /var/spool/cron/crontabs/$4
        chown $4:crontab /var/spool/cron/crontabs/$4
        chmod 600 /var/spool/cron/crontabs/$4
    fi

    if crontab -l -u $4 | grep -q "$2"; then
        (crontab -l -u $4 | sed "$3" | crontab -)
        (crontab -l -u $4 2>/dev/null; echo "$1") | crontab -
    else
        (crontab -l -u $4 2>/dev/null; echo "$1") | crontab -
    fi
}

# Function to safely write sudoers for a specific user
add_sudoers_for_user() {
  local user="$1"
  local file="/etc/sudoers.d/${user}-kiosk"

  echo "🔧 Creating sudoers file for $user..."

  cat <<EOF > "$file"
$user ALL=(ALL) NOPASSWD: /usr/local/bin/switchUser.sh, /opt/kiosk/[0-9]_*.sh
EOF

  chmod 440 "$file"
  chown root:root "$file"

  if visudo -cf "$file"; then
    echo "✅ Sudoers entry for $user is valid."
  else
    echo "❌ Invalid sudoers entry for $user. Removing it."
    rm -f "$file"
    exit 1
  fi
}

# Parse arguments
kioskuser=$1
adminuser=$2

SCRIPT=$(realpath "$0")
dir=$(dirname "$SCRIPT")
USER_HOME="/home/$kioskuser"
CONFIG_DIR="$USER_HOME/.config"
AUTOSTART_DIR="$CONFIG_DIR/autostart"

if [[ -z "$kioskuser" || -z "$adminuser" ]]; then
    echo "❌ Error: You must pass kioskuser and adminuser as arguments."
    exit 1
fi

section "Vytvoření kioskuser uživatele"
if id "$kioskuser" &>/dev/null; then
    echo "Uživatel $kioskuser již existuje. Přeskakuji vytvoření."
else
    adduser --disabled-password --gecos "" "$kioskuser"
    echo "kioskuser:kiosk" | chpasswd
fi
mkdir -p /home/$kioskuser/Desktop /home/$kioskuser/Downloads
chown -R $kioskuser:$kioskuser /home/$kioskuser

cp $dir/wallpapers/kiosk.png /home/$kioskuser/kiosk.png
chown $kioskuser:$kioskuser /home/$kioskuser/kiosk.png

section "APT update and install essentials"
apt update && apt upgrade -y
apt install -y openjdk-21-jre openjdk-21-jdk usbrelay unzip unclutter

section "Add kioskuser to dialout group"
adduser $kioskuser dialout

section "Add kioskuser to plugdev group"
adduser $kioskuser plugdev

section "Copy autostart .desktop files"
mkdir -p "$AUTOSTART_DIR"
cp "$dir/autostart/damePradloClient.desktop" "$AUTOSTART_DIR/"
cp "$dir/autostart/unclutter.desktop" "$AUTOSTART_DIR/"
chown $kioskuser:$kioskuser "$AUTOSTART_DIR/"*.desktop

section "Add coordinate transformation to .profile"
echo 'xinput set-prop "eGalax Inc. eGalaxTouch EXC3188-4633-09.00.00.00" --type=float "Coordinate Transformation Matrix" 0 1 0 -1 0 1 0 0 1' >> /home/$kioskuser/.profile
echo 'xinput set-prop "ILITEK       ILITEK-TOUCH                                                    " --type=float "Coordinate Transformation Matrix" 0 1 0 -1 0 1 0 0 1' >> /home/$kioskuser/.profile
echo 'xinput set-prop "eGalax Inc. eGalaxTouch EXC3188-4633-09.00.00.00" --type=float "Coordinate Transformation Matrix" 0 1 0 -1 0 1 0 0 1' >> /home/$adminuser/.profile
echo 'xinput set-prop "ILITEK       ILITEK-TOUCH                                                    " --type=float "Coordinate Transformation Matrix" 0 1 0 -1 0 1 0 0 1' >> /home/$adminuser/.profile

section "Install systemd timers for calibration"
mkdir -p /home/$adminuser/.config/systemd/user
cp $dir/calibrateService/calibrate-touch.service /home/$adminuser/.config/systemd/user/
cp $dir/calibrateService/calibrate-touch.timer /home/$adminuser/.config/systemd/user/
chown $adminuser:$adminuser /home/$adminuser/.config/systemd/user/calibrate-touch.*
systemctl --user daemon-reload
systemctl --user enable --now calibrate-touch.timer
systemctl --user enable --now calibrate-touch-2.timer

section "Disable IPv6"
sysctl -w net.ipv6.conf.all.disable_ipv6=1
sysctl -w net.ipv6.conf.default.disable_ipv6=1
sysctl -w net.ipv6.conf.lo.disable_ipv6=1
cp $dir/config/disable-ipv6.cfg /etc/default/grub.d/disable-ipv6.cfg

section "Setup KDE session for kioskuser"
sudo -u $kioskuser mkdir -p "$CONFIG_DIR" "$AUTOSTART_DIR" "$USER_HOME/.local/share"
chown -R $kioskuser:$kioskuser "$USER_HOME"

# KDE session config
section "Force KDE to start with an empty session"
echo -e "[General]\nloginMode=emptySession" > "$CONFIG_DIR/ksmserverrc"
chown $kioskuser:$kioskuser "$CONFIG_DIR/ksmserverrc"

section "Add switchUser script"
cp "$dir/switchUser/switchUser.sh" "/usr/local/bin/switchUser.sh"
cp "$dir/switchUser/switchUser.desktop" "/home/$adminuser/Desktop/switchUser.desktop"
cp "$dir/switchUser/switchUser.desktop" "/home/$kioskuser/Desktop/switchUser.desktop"
chown -R $adminuser:$adminuser /home/$adminuser/Desktop
chown -R $kioskuser:$kioskuser /home/$kioskuser/Desktop
chmod +x "/usr/local/bin/switchUser.sh"
chmod +x "/home/$adminuser/Desktop/switchUser.desktop"
chmod +x "/home/$kioskuser/Desktop/switchUser.desktop"

section "⚙️  Setting up sudoers entries for kioskuser..."
add_sudoers_for_user "kioskuser"

section "Odstraneni vychozi sddm konfigurace"
rm /etc/sddm.conf

section "✅ Hotovo"
read -p "Hotovo. Stiskni ENTER pro restartovani pocitace."
echo -e "[Autologin]\nUser=kioskuser\nSession=plasma" > /etc/sddm.conf.d/autologin.conf
sudo -u $kioskuser reboot
