#!/bin/bash

SCRIPT=$(realpath "$0")
dir=$(dirname "$SCRIPT")
ADMINUSER="adminuser"
KIOSKUSER="kioskuser"
LOGFILE="/var/log/kiosk-runme.log"

# Optional: Dry run mode
DRY_RUN=false
if [[ "${1:-}" == "--dry-run" ]]; then
    DRY_RUN=true
    echo "🚧 Dry run mode enabled. No changes will be made."
fi

# Logging setup
exec > >(tee -a "$LOGFILE") 2>&1

echo "📋 Starting runMe.sh at $(date)"

menuText="Vyber krok instalace\nZvol pomocí šipek nahoru a dolů, potvrď pomocí ENTER. Zmáčkni Ctrl+C pro zrušení."

find "$dir" -type f -name "*.sh" -exec chmod a+x {} \;

# Helper to run a step with error capture
run_step() {
    local script="$1"
    local title="$2"

    echo -e "\n▶️ Spouštím krok: $title"

    # Skip kioskuser check only for the first step
    if [[ "$script" != *1_configAdmin.sh ]] && [[ "$script" != *2_install.sh  ]]; then
        if ! id "$KIOSKUSER" &>/dev/null; then
            dialog --title "Chyba" --msgbox "Uživatel $KIOSKUSER neexistuje. Spusť nejprve krok pro vytvoření účtu." 10 50
            return 1
        fi
    fi

    if ! id "$ADMINUSER" &>/dev/null; then
        dialog --title "Chyba" --msgbox "Uživatel $ADMINUSER neexistuje. Zkontroluj instalaci systému." 10 50
        return 1
    fi

    if $DRY_RUN; then
        echo "(Dry run) Would execute: $script"
        return 0
    fi

    if ! pkexec "$script" "$KIOSKUSER" "$ADMINUSER" --keep-cwd; then
        dialog --title "Chyba: " --msgbox "Spuštění $script selhalo.\nZkontroluj log ve /var/log/kiosk-<stepName>.log" 10 50
        return 1
    fi
}

while true; do
    selectedStep=$(dialog --clear --no-cancel --no-tags --menu "$menuText" 50 150 150 \
        "1" "1) Konfigurace Admin - Konfigurace administratorskeho uctu" \
        "2" "2) Konfigrace Operačního systému - Zakladni konfigurace, Java, usbrelay, cronjob, autostart, ipv6, session" \
        "3" "3) TeamViewer - Instalace TeamViewer host aplikace" \
        "4" "4) Relé moduly - Konfigurace rele modulu" \
        "5" "5) USB zařízení - Konfigurace USB zařízení" \
        "6" "6) Instalace aplikace - Nainstaluje aplikaci" \
        "7" "7) Konfigurace aplikace - Průvodce konfigurací aplikace" \
        "8" "8) Finální konfigurace - Poslední konfigurační krok" \
        "9" "9) Konec - Ukončení instalační aplikace" \
        --output-fd 1)

    case $selectedStep in
        "1") run_step "$dir/1_configAdmin.sh" "Konfigurace administrátorského účtu" ;;
        "2") run_step "$dir/2_install.sh" "Konfigrace Operačního systému" ;;
        "3") run_step "$dir/3_teamViewer.sh" "Instalace TeamViewer" ;;
        "4") run_step "$dir/4_rele.sh" "Konfigurace relé modulů" ;;
        "5") run_step "$dir/5_usb.sh" "Konfigurace USB zařízení" ;;
        "6") run_step "$dir/6_installApp.sh" "Instalace aplikace" ;;
        "7") run_step "$dir/7_configApp.sh" "Konfigurace aplikace" ;;
        "8") run_step "$dir/8_finalConfig.sh" "Finální konfigurace" ;;
        "9") echo "👋 Ukončuji skript."; break;;
    esac

done

echo "✅ Dokončeno v $(date)"